package edu.mit.six825.bn.functiontable;

import java.util.Arrays;
import java.util.Iterator;



/**
 * An immutable set of FunctionVariables.
 * @author drayside
 */
public class FunctionVariableSet implements Comparable {
	private final FunctionVariable[] vars;
	
	/**
	 * construct a singleton set
	 * @param fv
	 */
	public FunctionVariableSet(final FunctionVariable fv) {
		this(new FunctionVariable[] {fv});
	}
	public FunctionVariableSet(final FunctionVariable[] v) {
		this(v, v.length);
	}
	public FunctionVariableSet(final FunctionVariable[] v, final int length) {
		if (!Utility.isSortedAndUnique(v, length)) 
			throw new IllegalArgumentException("should be sorted and unique");
		vars = new FunctionVariable[length];
		System.arraycopy(v, 0, vars, 0, length);
	}

	public int size() {
		return vars.length;
	}

	public Iterator assignmentIterator() {
		return new FunctionVariableSetAssignmentIterator();
	}

	/**
	 * @return
	 */
	public int cartesianProductSize() {
		int size = 1;
		for (int i = 0; i < vars.length; ++i) {
			size *= vars[i].domain.size();
		}
		return size;
	}
	
	/**
	 * This is a sort of ugly internal method: the place where the representation
	 * of both FunctionVariableSet and Assignment needs to be seen.
	 * So these classes are tightly coupled.
	 * @param values the array of values from an assignment -- not mutated
	 * @param v the variable to search for
	 * @return the value associated with v
	 */
	Comparable lookupVarInAssignment(final Comparable[] values, final FunctionVariable v) {
		final int index = Arrays.binarySearch(vars, v);
		if (index < 0) 
			return null;
		else return values[index];
	}
	
	public int compareTo(Object obj) {
		FunctionVariableSet other = (FunctionVariableSet)obj;
		return Utility.compareTwoArrays(vars, other.vars);
	}
	public boolean equals(Object obj) {
		if (obj instanceof FunctionVariableSet) {
			return (0 == compareTo(obj));
		} return false;
	}
	public int hashCode() {
		int hash = 1; // multiplicative identity
		for (int i = 0; i < vars.length; i++) {
			hash *= vars[i].hashCode();
		}
		return hash;
	}
	
	
	/**
	 * Iterates over all possible assignments for this FunctionVariableSet.
	 * This can be usefull for FunctionConstructors.
	 * @author drayside
	 */
	class FunctionVariableSetAssignmentIterator implements Iterator {

		private Assignment assignment = new Assignment(FunctionVariableSet.this);
		private int pos = 0;

		public boolean hasNext() {
			return (pos < cartesianProductSize());
		}

		public Object next() {
			final Assignment tmp = assignment;
			assignment = assignment.increment();
			pos++;
			return tmp;
		}

		public void remove() {
			throw new UnsupportedOperationException("can't remove from a space!");
		
		}
	}

	public static FunctionVariableSet union(final FunctionVariableSet s1, final FunctionVariableSet s2) {
		/*
		int l1 = s1.size();
		int l2 = s2.size();
		FunctionVariableSet U = s1;
		U.vars = new FunctionVariable[l1 + l2];
		int overkill = 0;
		for(int i = 0 ; i < l2 ; i++) {
			for(int j = 0 ; j < s2.size() ; j++) {
				if(s2.vars[i].equals(s1.vars[j]))
					overkill++;
				else
					U.vars[]
			}
		}
		*/

		final FunctionVariable[] tmp =  new FunctionVariable[s1.vars.length + s2.vars.length];
		int unionSize = 0;
		int i1 = 0;
		int i2 = 0;
		for (int i = 0; i < tmp.length; ++i) {
			 
			// still inside both s1 and s2			
			final int c = s1.vars[i1].compareTo(s2.vars[i2]);
			if (c == 0) {
				tmp[i] = s1.vars[i1];
				i1++;
				i2++;
			} else if (c < 0) {
				//current s1 less than current s2
				tmp[i] = s1.vars[i1++];
			} else if (c > 0) {
				//current s2 less than current s1
				tmp[i] = s2.vars[i2++];
			} 
			unionSize = i;

			// check if we've gone past s1 or s2
			if (i1 == s1.vars.length) {	
				// gone past the end of s1, copy the rest of s2
				final int extraLength = s2.vars.length-i2;
				System.arraycopy(s2.vars, i2, tmp, ++i, extraLength);
				unionSize = i + extraLength;
				break;
			}
			if (i2 == s2.vars.length) {	
				// gone past the end of s2, copy the rest of s1
				final int extraLength = s1.vars.length-i1;
				System.arraycopy(s1.vars, i1, tmp, ++i, extraLength);
				unionSize = i + extraLength;
				break;
			}

		}
		return new FunctionVariableSet(tmp, unionSize);
	}

	/**
	 * @param set
	 * @param variable
	 * @return
	 */
	public static FunctionVariableSet union(FunctionVariableSet set, FunctionVariable variable) {
		final int index = Arrays.binarySearch(set.vars, variable);
		if (index < 0) {
			// it's not in the set, so construct a new one with it
			final int insertionPoint = -1 * (index + 1);
			final FunctionVariable[] tmp = new FunctionVariable[set.vars.length+1];
			System.arraycopy(set.vars, 0, tmp, 0, insertionPoint);
			tmp[insertionPoint] = variable;
			// is the insertion point the last element?
			if (insertionPoint == tmp.length-1) 
				return new FunctionVariableSet(tmp);
			// still more stuff to copy
			System.arraycopy(set.vars, insertionPoint, tmp, insertionPoint+1, set.vars.length - insertionPoint);
			return new FunctionVariableSet(tmp);
		} else {
			// it is in the set
			// since these are immutable, we don't need to do anything.
			return set;
		}
	}
	
	public static FunctionVariableSet union(final Function[] functions) {
		// union all of the variable sets
		FunctionVariableSet v = functions[0].variables;
		for (int i = 1; i < functions.length; i++) {
			v = FunctionVariableSet.union(v, functions[i].variables);
		}
		return v;
	}

	public FunctionVariable getVariable(int i) {
		return vars[i];
	}
	
	public String toString() {
		return Utility.arrayToString(vars);
	}
	
	/**
	 * @param set
	 * @return
	 */
	public boolean isSubsetOf(final FunctionVariableSet superSet) {
		for (int i = 0; i < vars.length; ++i) {
			if (!superSet.contains(vars[i])) return false;
		}
		return true;
	}
	/**
	 * @param variable
	 * @return
	 */
	public boolean contains(final FunctionVariable variable) {
		final int index = Arrays.binarySearch(vars, variable);
		return (index >= 0);
	}

	/**
	 * @param set
	 * @return
	 */
	public boolean containsAnyOf(FunctionVariableSet set) {
		for (int i = 0; i < set.vars.length; i++) {
			if (contains(set.vars[i])) return true;		
		} 
		return false;
	}

	/**
	 * subtract the variables in set from this set
	 * @param set
	 * @return 
	 */
	public FunctionVariableSet subtract(final FunctionVariableSet set) {
		final FunctionVariable[] tmp = new FunctionVariable[vars.length];
		int j = 0;
		for (int i = 0; i < vars.length; i++) {
			if (!set.contains(vars[i])) tmp[j++] = vars[i];	
		}
		return new FunctionVariableSet(tmp, j);
	}
	/**
	 * @param v
	 * @return
	 */
	public FunctionVariableSet subtract(final FunctionVariable v) {
		return subtract(new FunctionVariableSet(v));
	}
	
}


