
package edu.mit.six825.bn.functiontable;

import java.util.Arrays;
import java.util.Iterator;



/**
 * A domain that a variable may vary over.  Domains are totally ordered.  Immutable.
 * @author drayside
 */
public class Domain {
	/**
	 * java.lang.Boolean is not Comparable, so we construct an anonymous
	 * sub-class to deal with this important special case.
	 */
	public final static Domain BOOLEAN_DOMAIN;
	static {
		Comparable[] b = new Comparable[2];
		b[0] = ComparableBoolean.FALSE;
		b[1] = ComparableBoolean.TRUE;
		BOOLEAN_DOMAIN = new Domain(b);
	}
	/*
	static {
		Comparable[] b = new Comparable[2];
		b[0] = new Integer(0); //Boolean.FALSE;
		b[1] = new Integer(1); //Boolean.TRUE;
		BOOLEAN_DOMAIN = new Domain(b) {
			public Object getValue(final DomainIndex i) {
				if (i.i == 0) return Boolean.FALSE; else return Boolean.TRUE;
			}
		};
	}
	*/
	private final Comparable[] values;
	
	/**
	 * memoized domain indices for this domain.
	 * entries instantiated when lazily: ie, when first required.
	 */
	private final DomainIndex[] indices;
	
	public Domain(final Comparable[] v) {
		if (v.length == 0) throw new IllegalArgumentException("a Domain must have at least one value in it!");
		this.values = new Comparable[v.length];
		System.arraycopy(v, 0, values, 0, v.length);
		Arrays.sort(values);
		indices = new DomainIndex[v.length];
	}
	
	public Comparable getValue(final DomainIndex i){
		if (!i.domain.equals(this)) throw new IllegalArgumentException("illegal DomainIndex " + i + " used on Domain " + toString());
		return values[i.i];
	}


	public Comparable getValue(int i) {
		return values[i];
	}	

	public int size() {
		return values.length;
	}
	
	public int hashCode() {
		return values[0].hashCode();
	}
	
	public boolean equals(Object obj) {
		if (obj instanceof Domain) {
			final Domain other = (Domain)obj;
			if (values.length != other.values.length) return false;
			for (int i = 0; i < values.length; ++i) {
				if (!values[i].equals(other.values[i])) return false;
			}
			return true;
		} else return false;
	}
	
	public String toString() {
		final StringBuffer b = new StringBuffer(values.length * 10);
		b.append("Domain:  <");
		for (int i = 0; i < values.length-1; ++i) {
			b.append(values[i].toString());
			b.append(", ");
		}
		b.append(values[values.length-1]);
		b.append(">");
		return b.toString();
	}
	
	public Iterator iterator() {
		// see if the index objects have been constructed yet
		// if not, construct and memoize
		constructIndices();
		// now we know we have the indices, just return them
		return new DomainIterator();
	}
	
	private class DomainIterator implements Iterator {
		private int i = 0;
		public boolean hasNext() {return (i < indices.length);}
		public Object next() { return indices[i++]; }
		public void remove() {throw new UnsupportedOperationException();}
	}
	/**
	 * @param comparable
	 * @return may return null if the value is not in this domain
	 */
	public DomainIndex getIndex(final Comparable val) {
		final int index = Arrays.binarySearch(values, val);
		if (index < 0) return null;
		constructIndices();
		return indices[index];
	}

	/**
	 * construct and memoize DomainIndices
	 */
	private void constructIndices() {
		if (indices[0] == null) {
			for (int i = 0; i < values.length; ++i) {
				indices[i] = new DomainIndex(this, i);
			}
		}	
	}

	/**
	 * @return the maximum value in this domain
	 */
	public Object getMaxValue() {
		return values[values.length-1];
	}
	/**
	 * @return the minimum value in this domain
	 */
	public Comparable getMinValue() {
		return values[0];
	}

	/**
	 * @param value
	 * @return the value after the given one, or null if at the last value already
	 */
	public Comparable getNextValue(final Comparable value) {
		final int index = Arrays.binarySearch(values, value);
		if (index >= 0) {
			if (index == values.length-1) return null;
			else return values[index+1];
		} else return null;
	}

}

