package edu.mit.six825.bn.functiontable;

import java.util.Iterator;

/**
 * Container for static methods that do computations with Functions.
 *
 * @author drayside
 */
public class Compute {

    /**
     * @return a new function with the same value in every entry, where the sum
     *         of all entries == 1.
     */
    public static Function makeConstantFunction(final FunctionVariable var) {
        return makeConstantFunction(new FunctionVariableSet(var));
    }

    /**
     * @return a new function with the same value in every entry, where the sum
     *         of all entries == 1.
     */
    public static Function makeConstantFunction(final FunctionVariableSet vars) {
        return makeConstantFunction(vars, 1.0d / ((double) vars.cartesianProductSize()));
    }

    /**
     * @return a new function with the same (given) value in every entry
     */
    public static Function makeConstantFunction(final FunctionVariableSet vars, final double value) {
        final double[] values = new double[vars.cartesianProductSize()];

        for (int i = 0; i < values.length; i++) {
            values[i] = value;
        }
        return new Function(vars, values);
    }

    public static Function makeBoolean(final FunctionVariable var, final double trueValue) {
        return makeBoolean(var, trueValue, 1.0d - trueValue);
    }

    public static Function makeBoolean(final FunctionVariable var, final double trueValue,
            final double falseValue) {

        if (!var.domain.equals(Domain.BOOLEAN_DOMAIN))
            throw new IllegalArgumentException(var + " is not over the ComparableBoolean domain");

        final double[] values = new double[2];
        values[0] = falseValue;
        values[1] = trueValue;
        return new Function(var, values);
    }

    public static Function makeBoolean(FunctionVariable var, double[] values) {
        return makeBoolean(new FunctionVariableSet(var), values);
    }

    public static Function makeBoolean(FunctionVariableSet vars, double[] values) {
        for (int i = 0; i < vars.size(); i++) {
            if (!vars.getVariable(i).domain.equals(Domain.BOOLEAN_DOMAIN))
                throw new IllegalArgumentException(vars.getVariable(i) + " is not over the ComparableBoolean domain");
        }
        return new Function(vars, values);
    }


    public static Function makeEvidenceFunction(final FunctionVariable var, final Assignment evidence) {
        return makeEvidenceFunction(new FunctionVariableSet(var), evidence);
    }

    public static Function makeEvidenceFunction(final FunctionVariableSet vars, final Assignment evidence) {

        final double[] values = new double[vars.cartesianProductSize()];

        for (final Iterator i = vars.assignmentIterator(); i.hasNext();) {

            final Assignment a = (Assignment) i.next();

            // evidence happened, other stuff didn't
            // this assignment is part of the evidence
            if (a.congruent(evidence))
                values[a.computePosition()] = 1.0d;
            // no, it's not: so it didn't happen
            else
                values[a.computePosition()] = 0.0d;
        }

        return new Function(vars, values);

    }

    public static Function normalize(final Function f) {
        final double sum = f.sum();
        final double[] values = new double[f.variables.cartesianProductSize()];

        for (final Iterator i = f.variables.assignmentIterator(); i.hasNext();) {
            final Assignment a = (Assignment) i.next();
            values[a.computePosition()] = f.evaluate(a) / sum;
        }
        return new Function(f.variables, values);
    }
}
