package edu.mit.six825.bn.functiontable;





/**
 * Maps variables to values in their domain.
 * Domains are expected to be small.
 * Backed by arrays and binary search.
 * Immutable.
 * @author drayside
 */
public class Assignment implements Comparable {

	public final FunctionVariableSet variables;
	private final Comparable[] values;

	/**
	 * assigns all vars to the minimum value of their respective domains
	 * @param vars
	 */
	public Assignment(FunctionVariableSet vars) {
		this.variables = vars;
		this.values = new Comparable[vars.size()];
		for (int i = 0; i < vars.size(); i++) {
			values[i] = vars.getVariable(i).domain.getMinValue();
		}
	}

	/**
	 *
	 * @param vars
	 * @param inValues may be longer than vars, in which case only the first |vars| items are taken
	 */
	public Assignment(final FunctionVariableSet vars, final Comparable[] inValues) {
		if (vars.size() > inValues.length) throw new IllegalArgumentException();
		this.variables = vars;
		this.values = new Comparable[vars.size()];
		System.arraycopy(inValues, 0, values, 0, vars.size());
	}


	/**
	 * The space of assignments is totally ordered, so a particular assignment
	 * can be specified by giving an index into that space.
	 * @param vars
	 * @param pos
	 */
	private Assignment(final FunctionVariableSet vars, final int pos) {
		this.variables = vars;
		this.values = computeValues(vars, pos);
	}

	/**
	 * the inverse of computePosition()
	 * @param vars
	 * @param pos
	 */
	private static Comparable[] computeValues(final FunctionVariableSet vars, final int pos) {
		return null;
	}

	/**
	 * Create a new assignment that is similar to the given assignment, except for the
	 * value associated with the given variable.
	 * @param a the base assignment
	 * @param variable a variable that's going to get a different value
	 * @param value the new value for the variable
	 */
        public Assignment(final Assignment a, final FunctionVariable variable,
                          final Comparable value) {
            variables = FunctionVariableSet.union(a.variables, variable);
            values = new Comparable[variables.size()];
            // now things are in a new order ...
            int j = 0;
            for (int i = 0; i < variables.size(); ++i) {
                if (variables.getVariable(i).equals(variable)) {
                    values[i] = value;
                    if (a.variables.contains(variable)) {
                        j++;
                    }
                } else {
                    values[i] = a.values[j++];
                }
            }
        }

	/**
	 * What happens when this gets to the end?  Wrap-around?
	 * @return a new Assignment
	 */
	public Assignment increment() {
		final Comparable[] incrementedValues = new Comparable[values.length];
		System.arraycopy(values, 0, incrementedValues, 0, values.length);
		// MSB is the last variable in the set: start there
		for (int i = variables.size()-1; i >=0; i--) {
			final FunctionVariable var = variables.getVariable(i);
			// do we have space to increment this variable?
			final Comparable assignedValue = getAssignedValue(var);
			if (assignedValue.equals(var.domain.getMaxValue())) {
				// no more space, more on to next variable
				// but this one has to go to the minimum value
				incrementedValues[i] = var.domain.getMinValue();
				continue;
			} else {
				// we've got space, so increment
				incrementedValues[i] = var.domain.getNextValue(values[i]);
				break;
			}
		}
		return new Assignment(variables, incrementedValues);
	}

	public int computePosition() {
		return computePosition(this, this.variables);
	}

	/**
	 * compute the position as if only variables were in this assignment
	 * @param variables a subset of the variables in this assignment
	 * @return
	 */
	public int computePosition(FunctionVariableSet otherVariables) {
		return computePosition(this, otherVariables);
	}

	private static int computePosition(final Assignment a, final FunctionVariableSet variables) {
		int index = 0;
		for (int i = 0; i < variables.size(); ++i) {
			final FunctionVariable var = variables.getVariable(i);
			final DomainIndex varValNdx = var.domain.getIndex(a.getAssignedValue(var));
			index = (index * var.domain.size()) + varValNdx.i;
		}
		return index;

	}

	/*
	public Assignment(final FunctionVariable[] inVars, final Comparable[] inValues) {
		if (inVars.length != inValues.length) throw new IllegalArgumentException("vars.length != values.length");
		// make a sorted copy of vars
		this.vars = new FunctionVariable[inVars.length];
		System.arraycopy(inVars, 0, this.vars, 0, inVars.length);
		Arrays.sort(vars);
		// make sure that the vars are unique
		for (int i = 0; i < vars.length-1; ++i) {
			if (vars[i].equals(vars[i+1])) throw new IllegalArgumentException("two equivalent vars in the same assignment! " + vars[i].name);
		}
		// now make sure we have the values in the right order
		this.values = new Comparable[inValues.length];
		for (int oldIndex = 0; oldIndex < inVars.length; ++oldIndex) {
			final int newIndex = Arrays.binarySearch(this.vars, inVars[oldIndex]);
			values[newIndex] = inValues[oldIndex];
		}
	}
	*/

	/**
	 * @param v
	 * @return the value assigned to variable <code>v</code>, or <code>null</code>
	 * if no value is assigned to <code>v</code> in this assignment.
	 */
	public Comparable getAssignedValue(FunctionVariable v) {
		return variables.lookupVarInAssignment(values, v);
	}
	public boolean contains(FunctionVariable v) {
		return (getAssignedValue(v) != null);
	}

	public int compareTo(final Object obj) {
		final Assignment other = (Assignment)obj;
		if (variables.equals(other.variables)) {
			 return Utility.compareTwoArrays(values, other.values);
		} else throw new RuntimeException("cannot compare Assignments with different vars");
	}
	public boolean equals(final Object obj) {
		return Utility.equals_basedOnCompareTo(this, obj);
	}
	public int hashCode() {
		return variables.hashCode();
	}
	public String toString() {
		StringBuffer b = new StringBuffer(20 * variables.size());
		b.append("<");
		b.append(variables.toString());
		b.append(" = ");
		b.append(Utility.arrayToString(values));
		b.append(">");
		return b.toString();
	}

	public Assignment subtract(final FunctionVariable v) {
		final FunctionVariable[] tmpVars = new FunctionVariable[variables.size()];
		final Comparable[] tmpVals = new Comparable[values.length];
		int j = 0;
		for (int i = 0; i < variables.size(); i++) {
			if (!v.equals(variables.getVariable(i))) {
				tmpVars[j] = variables.getVariable(i);
				tmpVals[j] = values[i];
				j++;
			}
		}
		return new Assignment(new FunctionVariableSet(tmpVars, j), tmpVals);
	}

	public Assignment subtract(Assignment a) {
		final FunctionVariable[] tmpVars = new FunctionVariable[variables.size()];
		final Comparable[] tmpVals = new Comparable[values.length];
		int j = 0;
		for (int i = 0; i < variables.size(); i++) {
			if (!a.contains(variables.getVariable(i))) {
				tmpVars[j] = variables.getVariable(i);
				tmpVals[j] = values[i];
				j++;
			}
		}
		return new Assignment(new FunctionVariableSet(tmpVars, j), tmpVals);
	}

	/**
	 * a.congruent(b) means that a.variables is a subset of b.variables, and all values agree
	 * @param fa
	 * @return
	 */
	public boolean congruent(final Assignment fa) {
		for (int i = 0; i < variables.size(); i++) {
			final FunctionVariable v = variables.getVariable(i);
			if (fa.contains(v)) {
				if (!values[i].equals(fa.getAssignedValue(v))) return false;
			}
		}
		return true;
	}

}

