package edu.mit.six825.bn.bayesnet;

import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;

import edu.mit.six825.bn.functiontable.FunctionVariable;
import edu.mit.six825.bn.functiontable.FunctionVariableSet;
import edu.mit.six825.bn.functiontable.Utility;


/**
 * A set of BayesNetNodes. Immutable.
 * 
 * @author drayside
 */
public class BayesNetNodeSet {

	public final static BayesNetNodeSet EMPTY_BAYES_NET_VARIABLE_SET = new BayesNetNodeSet();

	private final BayesNetNode[] nodes;

	public BayesNetNodeSet() {
		nodes = new BayesNetNode[0];
	}

	/**
	 * @param v1
	 * @param v2
	 */
	public BayesNetNodeSet(final BayesNetNode v1, final BayesNetNode v2) {
		if (v1.equals(v2))
			throw new IllegalArgumentException();
		nodes = new BayesNetNode[2];
		nodes[0] = v1;
		nodes[1] = v2;
	}

	public BayesNetNodeSet(final BayesNetNode[] inVars) {
		// ensure uniqueness ...
		if (!Utility.allElementsUnique(inVars))
			throw new IllegalArgumentException();
		this.nodes = new BayesNetNode[inVars.length];
		System.arraycopy(inVars, 0, this.nodes, 0, nodes.length);
		// sort them
		Arrays.sort(this.nodes);
	}

	/**
	 * Returns nodes in the net sorted in topological order from parents to
	 * children, i.e. if possible makes all parents show up before their
	 * children.
	 * 
	 * @param nodeName
	 * @return list of nodes
	 */
	public List getNodesWithTopologicalOrdering() {

		// get a list of all the nodes
		List retVal = new LinkedList();
		for (int i = 0; i < nodes.length; i++)
			retVal.add(nodes[i]);

		boolean swapped;
		do {
			swapped = false;
			ListIterator li = retVal.listIterator();
			while (li.hasNext()) {
				int currNdx = li.nextIndex();
				BayesNetNode curr = (BayesNetNode) li.next();

				BayesNetNodeSet currPar = curr.parents;
				for (int i = 0; i < currPar.nodes.length; i++) {
					int parNdx = retVal.indexOf(currPar.nodes[i]);
					if (parNdx > currNdx) {
						retVal.remove(currPar.nodes[i]);
						retVal.add(currNdx, currPar.nodes[i]);
						swapped = true;
					}
				}
				if (swapped) {
					// li has become invalid so break out and reenter
					break;
				}
			}

		} while (swapped == true);

		return retVal;
	}

	public FunctionVariableSet getFunctionVariableSet() {
		FunctionVariable[] fv = new FunctionVariable[nodes.length];
		for (int i = 0; i < nodes.length; ++i) {
			fv[i] = nodes[i].var;
		}
		return new FunctionVariableSet(fv);
	}

	public int size() {
		return nodes.length;
	}

	public Iterator iterator() {
		return new BayesNetNodeSetIterator();
	}

	private class BayesNetNodeSetIterator implements Iterator {
		private int index = 0;

		public boolean hasNext() {
			return (index < nodes.length);
		}

		public Object next() {
			return nodes[index++];
		}

		public void remove() {
			throw new UnsupportedOperationException();
		}
	}

	public BayesNetNode getNode(final int i) {
		return nodes[i];
	}

	public BayesNetNode getNode(final String name) {
		for (int i = 0; i < nodes.length; i++) {
			if (name.equals(nodes[i].var.name))
				return nodes[i];
		}
		return null;
	}

	public String toString() {
		return Utility.arrayToString(nodes);
	}

	/**
	 * @param other
	 * @return true if this contains other
	 */
	public boolean contains(final BayesNetNode other) {
		return (Arrays.binarySearch(nodes, other) >= 0);
	}

}